# SPDX-FileCopyrightText: Copyright (C) ARDUINO SRL (http://www.arduino.cc)
#
# SPDX-License-Identifier: MPL-2.0

import json
from datetime import datetime, UTC
from arduino.app_utils import *
from arduino.app_bricks.web_ui import WebUI

# 初始化Web UI
ui = WebUI()

# 小车状态
car_state = {
    "direction": "stop",  # stop, forward, backward, left, right
    "speed": 150,
    "last_update": None
}

# 触摸状态记录
touch_events = {}

def _iso_now() -> str:
    return datetime.now(UTC).isoformat()

def on_set_speed(sid, message):
    """设置速度"""
    try:
        speed = int(message)
        if 0 <= speed <= 255:
            Bridge.call("set_speed", speed)
            car_state["speed"] = speed
            car_state["last_update"] = _iso_now()
            print(f"[{_iso_now()}] Speed set to: {speed}")
            ui.send_message("car_state_update", car_state)
        else:
            ui.send_message("error", f"Invalid speed value: {speed}. Must be 0-255")
    except Exception as e:
        ui.send_message("error", f"Set speed error: {e}")

def on_move_forward(sid, message):
    """前进"""
    try:
        Bridge.call("move_forward")
        car_state["direction"] = "forward"
        car_state["last_update"] = _iso_now()
        print(f"[{_iso_now()}] Moving forward")
        ui.send_message("car_state_update", car_state)
    except Exception as e:
        ui.send_message("error", f"Move forward error: {e}")

def on_move_backward(sid, message):
    """后退"""
    try:
        Bridge.call("move_backward")
        car_state["direction"] = "backward"
        car_state["last_update"] = _iso_now()
        print(f"[{_iso_now()}] Moving backward")
        ui.send_message("car_state_update", car_state)
    except Exception as e:
        ui.send_message("error", f"Move backward error: {e}")

def on_turn_left(sid, message):
    """左转"""
    try:
        Bridge.call("turn_left")
        car_state["direction"] = "left"
        car_state["last_update"] = _iso_now()
        print(f"[{_iso_now()}] Turning left")
        ui.send_message("car_state_update", car_state)
    except Exception as e:
        ui.send_message("error", f"Turn left error: {e}")

def on_turn_right(sid, message):
    """右转"""
    try:
        Bridge.call("turn_right")
        car_state["direction"] = "right"
        car_state["last_update"] = _iso_now()
        print(f"[{_iso_now()}] Turning right")
        ui.send_message("car_state_update", car_state)
    except Exception as e:
        ui.send_message("error", f"Turn right error: {e}")

def on_stop_motors(sid, message):
    """停止"""
    try:
        Bridge.call("stop_motors")
        car_state["direction"] = "stop"
        car_state["last_update"] = _iso_now()
        print(f"[{_iso_now()}] Stopped")
        ui.send_message("car_state_update", car_state)
    except Exception as e:
        ui.send_message("error", f"Stop error: {e}")

def on_get_status():
    """获取状态"""
    try:
        status_code = Bridge.call("get_status")
        direction_code = status_code // 1000
        speed_value = status_code % 1000
        
        directions = ["stop", "forward", "backward", "left", "right"]
        car_state["direction"] = directions[direction_code] if direction_code < len(directions) else "unknown"
        car_state["speed"] = speed_value
        car_state["last_update"] = _iso_now()
    except Exception as e:
        print(f"Error getting status: {e}")
    
    return car_state

def on_touch_start(sid, message):
    """触摸开始事件（长按开始）"""
    try:
        data = json.loads(message) if isinstance(message, str) else message
        action = data.get("action")
        touch_id = data.get("id", sid)
        
        # 记录触摸开始
        touch_events[touch_id] = {
            "action": action,
            "start_time": datetime.now(UTC)
        }
        
        # 执行对应动作
        if action == "forward":
            on_move_forward(sid, None)
        elif action == "backward":
            on_move_backward(sid, None)
        elif action == "left":
            on_turn_left(sid, None)
        elif action == "right":
            on_turn_right(sid, None)
            
    except Exception as e:
        ui.send_message("error", f"Touch start error: {e}")

def on_touch_end(sid, message):
    """触摸结束事件（松开按钮）"""
    try:
        data = json.loads(message) if isinstance(message, str) else message
        touch_id = data.get("id", sid)
        
        # 移除触摸记录
        if touch_id in touch_events:
            action = touch_events[touch_id]["action"]
            duration = (datetime.now(UTC) - touch_events[touch_id]["start_time"]).total_seconds()
            del touch_events[touch_id]
            
            print(f"[{_iso_now()}] {action} held for {duration:.2f} seconds, stopping")
        
        # 停止小车
        on_stop_motors(sid, None)
        
    except Exception as e:
        ui.send_message("error", f"Touch end error: {e}")

# 注册消息处理器
ui.on_message("set_speed", on_set_speed)
ui.on_message("move_forward", on_move_forward)
ui.on_message("move_backward", on_move_backward)
ui.on_message("turn_left", on_turn_left)
ui.on_message("turn_right", on_turn_right)
ui.on_message("stop_motors", on_stop_motors)
ui.on_message("touch_start", on_touch_start)
ui.on_message("touch_end", on_touch_end)

# 暴露API
ui.expose_api("GET", "/status", on_get_status)

App.run()